define([
    'underscore',
    'backbone',
    'models/community-care-eligibility-model',
],
function(_, Backbone, CommunityCareEligModel) {
    'use strict';

    describe('Community Care Eligibility Model', function() {
        var communityCareEligModel;

        beforeEach(function() {
            communityCareEligModel = new CommunityCareEligModel({
                resourceName: 'model',
                vceCode: '123',
            });
        });

        it('can parse a Community Care Elig Model, without a vceCode in the response', function() {
            var response = {
                communityCareEligibilityInfo: {},
            };

            var communityCareEligModel2 = new CommunityCareEligModel({
                resourceName: 'model',
                vceCode: null,
            });
            communityCareEligModel2.parse(response);
            expect(communityCareEligModel2.get('vceCode')).toBeNull();
        });

        it('can parse a Community Care Elig Model, with a vceCode in the response', function() {
            var response = {
                communityCareEligibilityInfo: {
                    vceCode: 'M',
                },
            };

            communityCareEligModel.parse(response);

            expect(communityCareEligModel.get('vceCode')).toEqual('M');
            expect(communityCareEligModel.isCcEligible()).toBe(true);
        });

        it('can determine if the patient is CC eligible, for a valid vceCode', function() {
            var vceCodes = ['M', 'H', 'B', 'WT', 'MWT', 'HWT', 'X'];
            var vceCodesExpected = [true, true, false, false, true, true, false];
            var vceCodesActual = [];

            _.each(vceCodes, function(vceCode) {
                communityCareEligModel.set('vceCode', vceCode);
                vceCodesActual.push(communityCareEligModel.isCcEligible());
            });

            expect(vceCodesActual).toEqual(vceCodesExpected);
        });

        it('can determine if the patient is CC Eligible, for an invalid vceCode', function() {
            var eligible = communityCareEligModel.isCcEligible();

            expect(eligible).toBe(false);
        });
    });
});
